// Quantum Technology Landing Page JavaScript

document.addEventListener('DOMContentLoaded', function () {
    // Initialize all interactive elements
    initQuantumOrbit();
    initCounters();
    initProgressBars();
    initCharts();
    initScrollAnimations();
    initSmoothScrolling();
    initTimelineAnimations();

    initQuantumConcepts();
    initMobileMenu();
});

// Mobile Menu
function initMobileMenu() {
    const toggle = document.querySelector('.mobile-menu-toggle');
    const nav = document.querySelector('.nav-menu');
    const navLinks = document.querySelectorAll('.nav-menu a');

    if (toggle && nav) {
        toggle.addEventListener('click', () => {
            nav.classList.toggle('active');
            toggle.classList.toggle('active');
        });

        navLinks.forEach(link => {
            link.addEventListener('click', () => {
                nav.classList.remove('active');
                toggle.classList.remove('active');
            });
        });

        // Close menu when clicking outside
        document.addEventListener('click', (e) => {
            if (!nav.contains(e.target) && !toggle.contains(e.target) && nav.classList.contains('active')) {
                nav.classList.remove('active');
                toggle.classList.remove('active');
            }
        });
    }
}

// Quantum Orbit Animation
function initQuantumOrbit() {
    const canvas = document.getElementById('quantumOrbit');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2;

    class QuantumParticle {
        constructor(orbitRadius, speed, size, color) {
            this.orbitRadius = orbitRadius;
            this.angle = Math.random() * Math.PI * 2;
            this.speed = speed;
            this.size = size;
            this.color = color;
            this.x = centerX + Math.cos(this.angle) * this.orbitRadius;
            this.y = centerY + Math.sin(this.angle) * this.orbitRadius;
        }

        update() {
            this.angle += this.speed;
            this.x = centerX + Math.cos(this.angle) * this.orbitRadius;
            this.y = centerY + Math.sin(this.angle) * this.orbitRadius;
        }

        draw() {
            ctx.beginPath();
            ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2);
            ctx.fillStyle = this.color;
            ctx.fill();

            // Draw connection lines
            ctx.beginPath();
            ctx.moveTo(centerX, centerY);
            ctx.lineTo(this.x, this.y);
            ctx.strokeStyle = this.color + '30';
            ctx.lineWidth = 1;
            ctx.stroke();
        }
    }

    const particles = [
        new QuantumParticle(80, 0.02, 4, '#6366f1'),
        new QuantumParticle(120, 0.015, 3, '#8b5cf6'),
        new QuantumParticle(160, 0.01, 5, '#ec4899'),
        new QuantumParticle(100, 0.025, 2, '#6366f1'),
        new QuantumParticle(140, 0.008, 3, '#8b5cf6')
    ];

    function animate() {
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        // Draw center nucleus
        ctx.beginPath();
        ctx.arc(centerX, centerY, 8, 0, Math.PI * 2);
        ctx.fillStyle = '#6366f1';
        ctx.fill();

        // Draw orbit rings
        ctx.strokeStyle = 'rgba(99, 102, 241, 0.1)';
        ctx.lineWidth = 1;
        [80, 120, 160].forEach(radius => {
            ctx.beginPath();
            ctx.arc(centerX, centerY, radius, 0, Math.PI * 2);
            ctx.stroke();
        });

        // Update and draw particles
        particles.forEach(particle => {
            particle.update();
            particle.draw();
        });

        requestAnimationFrame(animate);
    }

    animate();
}

// Animated Counters
function initCounters() {
    const counters = document.querySelectorAll('.stat-number');
    const speed = 200;

    const animateCounter = (counter) => {
        const target = +counter.getAttribute('data-target');
        const increment = target / speed;

        const updateCount = () => {
            const count = +counter.innerText;

            if (count < target) {
                counter.innerText = Math.ceil(count + increment);
                setTimeout(updateCount, 10);
            } else {
                counter.innerText = target;
            }
        };

        updateCount();
    };

    const observerOptions = {
        threshold: 0.5
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                animateCounter(entry.target);
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);

    counters.forEach(counter => {
        observer.observe(counter);
    });
}

// Progress Bars Animation
function initProgressBars() {
    const progressBars = document.querySelectorAll('.progress-fill');

    const observerOptions = {
        threshold: 0.5
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const progress = entry.target.getAttribute('data-progress');
                entry.target.style.width = progress + '%';
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);

    progressBars.forEach(bar => {
        observer.observe(bar);
    });
}

// Charts Initialization
function initCharts() {
    // Market Growth Chart
    const marketCtx = document.getElementById('marketChart');
    if (marketCtx) {
        new Chart(marketCtx, {
            type: 'line',
            data: {
                labels: ['2020', '2021', '2022', '2023', '2024', '2025', '2030'],
                datasets: [{
                    label: '시장 규모 (십억 달러)',
                    data: [2.5, 4.2, 6.8, 10.5, 15.3, 22.1, 175],
                    borderColor: '#6366f1',
                    backgroundColor: 'rgba(99, 102, 241, 0.1)',
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        labels: {
                            color: '#f1f5f9'
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(148, 163, 184, 0.1)'
                        },
                        ticks: {
                            color: '#94a3b8'
                        }
                    },
                    x: {
                        grid: {
                            color: 'rgba(148, 163, 184, 0.1)'
                        },
                        ticks: {
                            color: '#94a3b8'
                        }
                    }
                }
            }
        });
    }

    // Investment Distribution Chart
    const investmentCtx = document.getElementById('investmentChart');
    if (investmentCtx) {
        new Chart(investmentCtx, {
            type: 'doughnut',
            data: {
                labels: ['컴퓨팅', '센싱', '통신', '헬스케어', '기타'],
                datasets: [{
                    data: [35, 20, 15, 20, 10],
                    backgroundColor: [
                        '#6366f1',
                        '#8b5cf6',
                        '#ec4899',
                        '#f59e0b',
                        '#10b981'
                    ],
                    borderWidth: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            color: '#f1f5f9',
                            padding: 20
                        }
                    }
                }
            }
        });
    }

    // Regional Leadership Chart
    const regionalCtx = document.getElementById('regionalChart');
    if (regionalCtx) {
        new Chart(regionalCtx, {
            type: 'bar',
            data: {
                labels: ['북미', '유럽', '아시아 태평양', '중국', '기타'],
                datasets: [{
                    label: '양자 리더십 지수',
                    data: [85, 72, 68, 65, 35],
                    backgroundColor: [
                        'rgba(99, 102, 241, 0.8)',
                        'rgba(139, 92, 246, 0.8)',
                        'rgba(236, 72, 153, 0.8)',
                        'rgba(245, 158, 11, 0.8)',
                        'rgba(16, 185, 129, 0.8)'
                    ],
                    borderWidth: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        labels: {
                            color: '#f1f5f9'
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(148, 163, 184, 0.1)'
                        },
                        ticks: {
                            color: '#94a3b8'
                        }
                    },
                    x: {
                        grid: {
                            color: 'rgba(148, 163, 184, 0.1)'
                        },
                        ticks: {
                            color: '#94a3b8'
                        }
                    }
                }
            }
        });
    }

    // Technology Adoption Chart
    const adoptionCtx = document.getElementById('adoptionChart');
    if (adoptionCtx) {
        new Chart(adoptionCtx, {
            type: 'radar',
            data: {
                labels: ['양자 컴퓨팅', '양자 센싱', '양자 통신', '양자 암호화', '양자 시뮬레이션'],
                datasets: [{
                    label: '현재 도입률',
                    data: [45, 60, 35, 50, 40],
                    borderColor: '#6366f1',
                    backgroundColor: 'rgba(99, 102, 241, 0.2)',
                    pointBackgroundColor: '#6366f1',
                    pointBorderColor: '#fff',
                    pointHoverBackgroundColor: '#fff',
                    pointHoverBorderColor: '#6366f1'
                }, {
                    label: '2030년 예측',
                    data: [85, 90, 80, 95, 88],
                    borderColor: '#ec4899',
                    backgroundColor: 'rgba(236, 72, 153, 0.2)',
                    pointBackgroundColor: '#ec4899',
                    pointBorderColor: '#fff',
                    pointHoverBackgroundColor: '#fff',
                    pointHoverBorderColor: '#ec4899'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        labels: {
                            color: '#f1f5f9'
                        }
                    }
                },
                scales: {
                    r: {
                        beginAtZero: true,
                        max: 100,
                        grid: {
                            color: 'rgba(148, 163, 184, 0.1)'
                        },
                        pointLabels: {
                            color: '#94a3b8'
                        },
                        ticks: {
                            color: '#94a3b8',
                            backdropColor: 'transparent'
                        }
                    }
                }
            }
        });
    }
}

// Scroll Animations
function initScrollAnimations() {
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('visible');
            }
        });
    }, observerOptions);

    // Observe elements with scroll-animate class
    const scrollElements = document.querySelectorAll('.scroll-animate, .infographic-card, .chart-container');
    scrollElements.forEach(el => {
        el.classList.add('scroll-animate');
        observer.observe(el);
    });
}

// Smooth Scrolling
function initSmoothScrolling() {
    const links = document.querySelectorAll('a[href^="#"]');

    links.forEach(link => {
        link.addEventListener('click', function (e) {
            e.preventDefault();

            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);

            if (targetElement) {
                const offsetTop = targetElement.offsetTop - 80;

                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        });
    });
}

// Timeline Animations
function initTimelineAnimations() {
    const timelineItems = document.querySelectorAll('.timeline-item');

    const observerOptions = {
        threshold: 0.3
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('visible');
            }
        });
    }, observerOptions);

    timelineItems.forEach(item => {
        observer.observe(item);
    });
}

// Interactive Card Effects
document.addEventListener('DOMContentLoaded', function () {
    const cards = document.querySelectorAll('.infographic-card');

    cards.forEach(card => {
        card.addEventListener('mouseenter', function () {
            this.style.transform = 'translateY(-10px) scale(1.02)';
        });

        card.addEventListener('mouseleave', function () {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });
});

// Parallax Effect for Hero Section
window.addEventListener('scroll', function () {
    const scrolled = window.pageYOffset;
    const hero = document.querySelector('.hero');
    const heroContent = document.querySelector('.hero-content');

    if (hero && heroContent) {
        hero.style.transform = `translateY(${scrolled * 0.5}px)`;
        heroContent.style.transform = `translateY(${scrolled * 0.3}px)`;
        heroContent.style.opacity = 1 - scrolled / 600;
    }
});

// Dynamic Year Update
document.addEventListener('DOMContentLoaded', function () {
    const yearElements = document.querySelectorAll('[data-year]');
    const currentYear = new Date().getFullYear();

    yearElements.forEach(element => {
        const year = element.getAttribute('data-year');
        if (parseInt(year) <= currentYear) {
            element.style.opacity = '1';
        } else {
            element.style.opacity = '0.6';
        }
    });
});

// Quantum Concepts Visualizations
function initQuantumConcepts() {
    initSuperpositionVisualization();
    initEntanglementVisualization();
    initMeasurementVisualization();
    initGatesVisualization();
}

// Superposition Visualization
function initSuperpositionVisualization() {
    const canvas = document.getElementById('superpositionCanvas');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2;

    let phase = 0;

    function drawSuperposition() {
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        // Draw Bloch sphere representation
        ctx.strokeStyle = 'rgba(99, 102, 241, 0.3)';
        ctx.lineWidth = 1;

        // Draw sphere circles
        ctx.beginPath();
        ctx.ellipse(centerX, centerY, 60, 60, 0, 0, Math.PI * 2);
        ctx.stroke();

        ctx.beginPath();
        ctx.ellipse(centerX, centerY, 60, 20, 0, 0, Math.PI * 2);
        ctx.stroke();

        ctx.beginPath();
        ctx.ellipse(centerX, centerY, 20, 60, 0, 0, Math.PI * 2);
        ctx.stroke();

        // Draw state vector
        const theta = Math.PI / 4;
        const phi = phase;
        const x = centerX + 50 * Math.sin(theta) * Math.cos(phi);
        const y = centerY - 50 * Math.cos(theta);
        const z = 50 * Math.sin(theta) * Math.sin(phi);

        ctx.strokeStyle = '#6366f1';
        ctx.lineWidth = 3;
        ctx.beginPath();
        ctx.moveTo(centerX, centerY);
        ctx.lineTo(x, y);
        ctx.stroke();

        // Draw arrow head
        ctx.fillStyle = '#6366f1';
        ctx.beginPath();
        ctx.arc(x, y, 4, 0, Math.PI * 2);
        ctx.fill();

        // Draw probability amplitudes
        const alpha = Math.cos(theta / 2);
        const beta = Math.sin(theta / 2);

        ctx.fillStyle = '#ec4899';
        ctx.font = '12px Noto Sans KR';
        ctx.fillText(`|0⟩: ${alpha.toFixed(2)}`, 10, 20);
        ctx.fillText(`|1⟩: ${beta.toFixed(2)}`, 10, 35);

        phase += 0.02;
        requestAnimationFrame(drawSuperposition);
    }

    drawSuperposition();
}

// Entanglement Visualization
function initEntanglementVisualization() {
    const canvas = document.getElementById('entanglementCanvas');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2;

    let time = 0;

    function drawEntanglement() {
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        // Draw two entangled particles
        const particle1X = centerX - 40;
        const particle2X = centerX + 40;
        const particleY = centerY;

        // Draw entanglement connection
        ctx.strokeStyle = 'rgba(236, 72, 153, 0.5)';
        ctx.lineWidth = 2;
        ctx.setLineDash([5, 5]);
        ctx.beginPath();
        ctx.moveTo(particle1X, particleY);
        ctx.lineTo(particle2X, particleY);
        ctx.stroke();
        ctx.setLineDash([]);

        // Draw particles with correlated states
        const phase = Math.sin(time) * Math.PI;
        const radius = 8 + Math.sin(time * 2) * 2;

        // Particle 1
        ctx.fillStyle = '#6366f1';
        ctx.beginPath();
        ctx.arc(particle1X, particleY, radius, 0, Math.PI * 2);
        ctx.fill();

        // Particle 2 (correlated state)
        ctx.fillStyle = '#ec4899';
        ctx.beginPath();
        ctx.arc(particle2X, particleY, radius, 0, Math.PI * 2);
        ctx.fill();

        // Draw correlation indicator
        ctx.fillStyle = '#94a3b8';
        ctx.font = '10px Noto Sans KR';
        ctx.fillText('상관관계', centerX - 25, centerY + 30);

        // Draw Bell state notation
        ctx.fillStyle = '#8b5cf6';
        ctx.font = '11px Inter';
        ctx.fillText('|Φ⁺⟩', centerX - 15, 15);

        time += 0.03;
        requestAnimationFrame(drawEntanglement);
    }

    drawEntanglement();
}

// Measurement Visualization
function initMeasurementVisualization() {
    const canvas = document.getElementById('measurementCanvas');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2;

    let collapseProgress = 0;
    let isCollapsing = false;

    function drawMeasurement() {
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        // Draw superposition state before collapse
        if (collapseProgress < 1) {
            // Draw probability cloud
            const gradient = ctx.createRadialGradient(centerX, centerY, 0, centerX, centerY, 40);
            gradient.addColorStop(0, 'rgba(99, 102, 241, 0.8)');
            gradient.addColorStop(1, 'rgba(99, 102, 241, 0.1)');

            ctx.fillStyle = gradient;
            ctx.beginPath();
            ctx.arc(centerX, centerY, 40 * (1 - collapseProgress * 0.5), 0, Math.PI * 2);
            ctx.fill();

            // Draw probability amplitudes
            ctx.fillStyle = '#94a3b8';
            ctx.font = '10px Noto Sans KR';
            ctx.fillText('중첩 상태', centerX - 30, centerY - 50);
        }

        // Draw collapsed state
        if (collapseProgress > 0) {
            const finalX = centerX + (Math.random() - 0.5) * 40;
            const finalY = centerY + (Math.random() - 0.5) * 40;
            const currentX = centerX + (finalX - centerX) * collapseProgress;
            const currentY = centerY + (finalY - centerY) * collapseProgress;

            ctx.fillStyle = '#ec4899';
            ctx.beginPath();
            ctx.arc(currentX, currentY, 8, 0, Math.PI * 2);
            ctx.fill();

            if (collapseProgress >= 1) {
                ctx.fillStyle = '#94a3b8';
                ctx.font = '10px Noto Sans KR';
                ctx.fillText('측정됨', currentX - 20, currentY + 20);
            }
        }

        // Auto-collapse animation
        if (!isCollapsing && Math.random() < 0.01) {
            isCollapsing = true;
        }

        if (isCollapsing) {
            collapseProgress += 0.02;
            if (collapseProgress >= 1) {
                collapseProgress = 0;
                isCollapsing = false;
            }
        }

        requestAnimationFrame(drawMeasurement);
    }

    drawMeasurement();
}

// Quantum Gates Visualization
function initGatesVisualization() {
    const canvas = document.getElementById('gatesCanvas');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2;

    let gatePhase = 0;

    function drawGates() {
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        // Draw quantum circuit
        ctx.strokeStyle = 'rgba(148, 163, 184, 0.5)';
        ctx.lineWidth = 1;

        // Draw qubit lines
        for (let i = 0; i < 2; i++) {
            const y = centerY - 20 + i * 40;
            ctx.beginPath();
            ctx.moveTo(20, y);
            ctx.lineTo(canvas.width - 20, y);
            ctx.stroke();
        }

        // Draw Hadamard gate
        const gateX = 60 + Math.sin(gatePhase) * 10;
        ctx.fillStyle = '#6366f1';
        ctx.fillRect(gateX - 15, centerY - 35, 30, 30);
        ctx.fillStyle = 'white';
        ctx.font = 'bold 14px Inter';
        ctx.fillText('H', gateX - 5, centerY - 15);

        // Draw CNOT gate
        const cnotX = 120 + Math.sin(gatePhase + Math.PI) * 10;
        ctx.fillStyle = '#ec4899';
        ctx.beginPath();
        ctx.arc(cnotX, centerY - 20, 8, 0, Math.PI * 2);
        ctx.fill();

        ctx.beginPath();
        ctx.arc(cnotX, centerY + 20, 8, 0, Math.PI * 2);
        ctx.fill();

        ctx.strokeStyle = '#ec4899';
        ctx.lineWidth = 2;
        ctx.beginPath();
        ctx.moveTo(cnotX, centerY - 12);
        ctx.lineTo(cnotX, centerY + 12);
        ctx.stroke();

        // Draw output states
        ctx.fillStyle = '#8b5cf6';
        ctx.font = '10px Noto Sans KR';
        ctx.fillText('|ψ⟩', 25, centerY - 17);
        ctx.fillText('|0⟩', 25, centerY + 23);

        ctx.fillText('(H|ψ⟩)', cnotX + 20, centerY - 17);
        ctx.fillText('CNOT', cnotX + 20, centerY + 23);

        gatePhase += 0.02;
        requestAnimationFrame(drawGates);
    }

    drawGates();
}

// Add hover effect to navigation items
document.addEventListener('DOMContentLoaded', function () {
    const navItems = document.querySelectorAll('.nav-menu a');

    navItems.forEach(item => {
        item.addEventListener('mouseenter', function () {
            this.style.transform = 'translateY(-2px)';
        });

        item.addEventListener('mouseleave', function () {
            this.style.transform = 'translateY(0)';
        });
    });
});